/*
    This file is part of g15-sysstats.

    g15-sysstats is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    g15-sysstats is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with g15-sysstats; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    (c) 2008, 2009 Tobias Baeumer (TobiasBaeumer@gmail.com)
 */

#define TTF_SUPPORT 1
#define MAX_LINE_LEN 254
#define GOOD 0
#define BAD -1
#define UGLY -2
#define LEFT 0
#define CENTER 1
#define RIGHT 2

#include <iostream>
#include <statgrab.h>
#include <stdio.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <math.h>
#include <pthread.h>
#include <libg15.h>
#include <g15daemon_client.h>
#include <libg15render.h>

char CPUCORECOUNT[MAX_LINE_LEN], DRIVE1[MAX_LINE_LEN], DRIVE2[MAX_LINE_LEN], NETIF[MAX_LINE_LEN], LCD_FONT_SIZE_MEDIUM[MAX_LINE_LEN], LCD_FONT_SIZE_SMALL[MAX_LINE_LEN], LCD_FONT_MEDIUM[MAX_LINE_LEN], LCD_FONT_SMALL[MAX_LINE_LEN], MAX_UPSTREAM[MAX_LINE_LEN], MAX_DOWNSTREAM[MAX_LINE_LEN], CASETEMPCOMMAND[MAX_LINE_LEN], CPUTEMPCOMMAND[MAX_LINE_LEN];
int const NUM_BYTES = G15_LCD_WIDTH * G15_LCD_HEIGHT / 8;
int cycle;
unsigned char test_data[NUM_BYTES];
extern short logo_data[];
int g15screen_fd = 0;
g15canvas *canvas;
pthread_t gfxthread, clockthread, tempthread, loadthread, netthread, cputhread, memthread, spcthread, keywatch;
pthread_mutex_t LCDupdateMutex = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t LCDdrawMutex = PTHREAD_MUTEX_INITIALIZER;
using namespace std;

int read_config_var( char *, char * , char [] );
int __nsleep(const struct timespec *req, struct timespec *rem);
int msleep(unsigned long milisec);

void
updateScreen (void)
{
	pthread_mutex_lock( &LCDupdateMutex );
	//pthread_mutex_lock( &LCDdrawMutex );
	if (!canvas->mode_cache) { g15_send (g15screen_fd, (char *) canvas->buffer, G15_BUFFER_LEN); }
	//pthread_mutex_unlock( &LCDdrawMutex );
	pthread_mutex_unlock( &LCDupdateMutex );

}

void
clearArea (int myx1, int myy1, int myx2, int myy2)
{
	g15r_drawBar(canvas, myx1, myy1, myx2, myy2, G15_PIXEL_NOFILL, 100, 100, 0); // crazy method to clear the area
	updateScreen();
}

void
connectToScreen (void)
{
  if ((g15screen_fd = new_g15_screen (G15_G15RBUF)) < 0)
    {
      cerr << "Sorry, can't connect to the G15daemon" << endl;
      exit (-1);
    }

  canvas = (g15canvas *) malloc (sizeof (g15canvas));

  g15r_initCanvas (canvas);
}

void
*DrawGraphics (void *ptr)
{
	int width = 160, height = 43;
	const char *buf = g15r_loadWbmpToBuf((char *)"./sirreal.wbmp", &width, &height);
	//while(1) {
		pthread_mutex_lock( &LCDdrawMutex );
		g15r_drawSprite(canvas, (char *)buf, 0, 0, width, height, 0, 0, width);
		//somehow the last line x&y got stolen so we draw it by our self
		g15r_drawLine(canvas, 159, 8, 159, 12, G15_PIXEL_FILL); 	//y
		g15r_drawLine(canvas, 159, 14, 159, 18, G15_PIXEL_FILL);	//y
		g15r_drawLine(canvas, 159, 22, 159, 43, G15_PIXEL_FILL);	//y

		g15r_setPixel(canvas, 0, 42, G15_PIXEL_FILL);				//x
		g15r_setPixel(canvas, 38, 42, G15_PIXEL_FILL);				//x
		g15r_setPixel(canvas, 40, 42, G15_PIXEL_FILL);				//x
		g15r_setPixel(canvas, 78, 42, G15_PIXEL_FILL);				//x
		g15r_setPixel(canvas, 81, 42, G15_PIXEL_FILL);				//x
		g15r_setPixel(canvas, 119, 42, G15_PIXEL_FILL);				//x
		g15r_setPixel(canvas, 121, 42, G15_PIXEL_FILL);				//x
		updateScreen();
		pthread_mutex_unlock( &LCDdrawMutex );
		//sleep(60);
	//}
	return (void*)NULL;
}

void
*DrawClock (void *ptr)
{
	time_t zeit;
	char strzeit[10], strdatum[10];
	struct tm *zeitinfo;
	setlocale(LC_TIME, "de_DE");

	while(1) {
		pthread_mutex_lock( &LCDdrawMutex );
		clearArea(0, 0, 160, 5);
		time(&zeit);
		zeitinfo = localtime(&zeit);
		strftime(strzeit, 9, "%T", zeitinfo);
		strftime(strdatum, 9, "%d.%m.%y", zeitinfo);
		g15r_ttfPrint(canvas, 0, 0, atoi(LCD_FONT_SIZE_MEDIUM), 0, G15_COLOR_BLACK, LEFT, strzeit);
		g15r_ttfPrint(canvas, 0, 0, atoi(LCD_FONT_SIZE_MEDIUM), 0, G15_COLOR_BLACK, RIGHT, strdatum);
		updateScreen();
		pthread_mutex_unlock( &LCDdrawMutex );
		msleep(950);
	}
	return (void*)NULL;
}

void
*DrawTemps (void *ptr)
{
	FILE *fpipe;
	char line[MAX_LINE_LEN];
	char caseOut[MAX_LINE_LEN], cpuOut[MAX_LINE_LEN];

	while(1) {
		if ( !(fpipe = (FILE*)popen(CASETEMPCOMMAND,"r")) ) { printf("\tProblems with pipe for getting case temp"); }
		if ( fscanf(fpipe, "%s", line) ) { sprintf(caseOut, "%sC", line); }
		pclose(fpipe);

		if ( !(fpipe = (FILE*)popen(CPUTEMPCOMMAND,"r")) ) { printf("\tProblems with pipe for getting cpu temp"); }
		if ( fscanf(fpipe, "%s", line) ) { sprintf(cpuOut, "%sC", line); }
		pclose(fpipe);

		pthread_mutex_lock( &LCDdrawMutex );
		clearArea(123, 28, 158, 43);
		g15r_ttfPrint(canvas, 123, 29, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, LEFT, (char *)"cpu0:");
		g15r_ttfPrint(canvas, 123, 36, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, LEFT, (char *)"cpu1:");
		g15r_ttfPrint(canvas, 145, 29, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, LEFT, caseOut);
		g15r_ttfPrint(canvas, 145, 36, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, LEFT, cpuOut);
		updateScreen();
		pthread_mutex_unlock( &LCDdrawMutex );
		sleep(15);
	}
	return (void*)NULL;
}

void
*DrawNetworkTraffic (void *ptr)
{
	int i, num_network_stats, maxOutKB, maxInKB;
	char rxOut[MAX_LINE_LEN], txOut[MAX_LINE_LEN];
	sg_network_io_stats *network_stats;
	network_stats = sg_get_network_io_stats_diff(&num_network_stats);

	while(1) {
		network_stats = sg_get_network_io_stats_diff(&num_network_stats);
		for(i = 0; i < num_network_stats; i++) {
			if( (strcmp(network_stats->interface_name, NETIF) != 0) ) {
				network_stats++;
			} else {
				sprintf(rxOut, "%5lld                       ", (network_stats->rx / 1024 * 2));
				sprintf(txOut, "%5lld                       ", (network_stats->tx / 1024 * 2));
				pthread_mutex_lock( &LCDdrawMutex );
				clearArea(42, 28, 77, 43);
				g15r_ttfPrint(canvas, 59, 22, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, LEFT, network_stats->interface_name);
				g15r_ttfPrint(canvas, 42, 29, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, LEFT, (char *)"<-");
				g15r_ttfPrint(canvas, 43, 36, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, LEFT, (char *)"->");
				g15r_ttfPrint(canvas, 68, 29, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, LEFT, (char *)"kb");
				g15r_ttfPrint(canvas, 68, 36, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, LEFT, (char *)"kb");
				g15r_ttfPrint(canvas, 44, 29, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, RIGHT, rxOut);
				g15r_ttfPrint(canvas, 44, 36, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, RIGHT, txOut);

				maxOutKB = atoi(MAX_UPSTREAM) / 8;
				maxInKB = atoi(MAX_DOWNSTREAM) / 8;
				//cout << "max down kb/s: " << maxInKB << " max out kb/s: " << maxOutKB << endl;
				clearArea(101, 9, 158, 11);
				clearArea(101, 15, 158, 17);
				g15r_drawBar(canvas, 101, 9, 158, 11, G15_PIXEL_FILL, network_stats->rx / 1024 * 2, maxInKB, 0);
				g15r_drawBar(canvas, 101, 15, 158, 17, G15_PIXEL_FILL, network_stats->tx / 1024 * 2, maxOutKB, 0);
				updateScreen();
				pthread_mutex_unlock( &LCDdrawMutex );
			}
		}
		msleep(500);
	}
	return (void*)NULL;
}

void
*DrawCPUStats (void *ptr)
{
	int corecount, whatever;
	struct stat st;
	FILE* kernelStats;
	if(stat("/sys/devices/system/cpu/cpu2",&st) == 0) { corecount = 3; }
	else { // no 3rd cpu found
		if(stat("/sys/devices/system/cpu/cpu1",&st) == 0) { corecount = 2; }
		else { corecount = 1; } // no 2nd cpu found, assume singlecore machine
	}

	int cpu0[5] = {0,0,0,0,0}, cpu1[5] = {0,0,0,0,0}, cpu2[5] = {0,0,0,0,0};
	int cpu0old[5] = {0,0,0,0,0}, cpu1old[5] = {0,0,0,0,0}, cpu2old[5] = {0,0,0,0,0};
	int targetpx, targetpx1, targetpxa;
	float totalusage, user, nice, system, idle, calc;
	float totalusage1, user1, nice1, system1, idle1, calc1;
	float totalusagea, usera, nicea, systema, idlea, calca;
	char line [MAX_LINE_LEN];
	int linenr = 0;

	while(1) {
		pthread_mutex_lock( &LCDdrawMutex );
		if( (kernelStats = fopen("/proc/stat", "r")) != NULL ) {
			setbuf(kernelStats, NULL);
			linenr = 0;
			while ( fgets ( line, sizeof line, kernelStats ) != NULL ) /* read a line */
			{
				if(linenr > 0 && linenr <= corecount) {
					switch(linenr) {
					case 1:
						cpu0old[0] = cpu0[0];
						cpu0old[1] = cpu0[1];
						cpu0old[2] = cpu0[2];
						cpu0old[3] = cpu0[3];
						cpu0old[4] = cpu0[4];
						sscanf(line, "cpu%d %d %d %d %d", &whatever, &cpu0[0], &cpu0[1], &cpu0[2], &cpu0[3]);
						cpu0[4] = cpu0[0] + cpu0[1] + cpu0[2];
						break;
					case 2:
						cpu1old[0] = cpu1[0];
						cpu1old[1] = cpu1[1];
						cpu1old[2] = cpu1[2];
						cpu1old[3] = cpu1[3];
						cpu1old[4] = cpu1[4];
						sscanf(line, "cpu%d %d %d %d %d", &whatever, &cpu1[0], &cpu1[1], &cpu1[2], &cpu1[3]);
						cpu1[4] = cpu1[0] + cpu1[1] + cpu1[2];
						break;
					case 3:
						cpu2old[0] = cpu2[0];
						cpu2old[1] = cpu2[1];
						cpu2old[2] = cpu2[2];
						cpu2old[3] = cpu2[3];
						cpu2old[4] = cpu2[4];
						sscanf(line, "cpu%d %d %d %d %d", &whatever, &cpu0[0], &cpu2[1], &cpu2[2], &cpu2[3]);
						cpu2[4] = cpu2[0] + cpu2[1] + cpu2[2];
						break;
					}
				}
				linenr++;
			}
			fclose ( kernelStats );
		}

		clearArea(20, 9, 76, 11);
		switch(corecount) {
		case 1:
			user = cpu0[0] - cpu0old[0];
			nice = cpu0[1] - cpu0old[1];
			system = cpu0[2] - cpu0old[2];
			idle = cpu0[3] - cpu0old[3];
			totalusage = user + nice + system;
			calc = 100 - (idle / (totalusage + idle) * 100);
			//cout << "Formula: 100 - (" << idle << " / (" << totalusage << " + " << idle << ") * 100)" << endl;
			//g15r_drawBar(canvas, 20, 9, 76, 11, G15_PIXEL_FILL, ceil(calc), 100, 0);
			targetpx = ceil(56 * calc/100)+20;
			//cout << "Result: " << calc << " Targetpx: " << targetpx << endl;
			g15r_drawLine(canvas, 20, 9, targetpx, 9, G15_PIXEL_FILL);
			g15r_drawLine(canvas, 20, 10, targetpx, 10, G15_PIXEL_FILL);
			g15r_drawLine(canvas, 20, 11, targetpx, 11, G15_PIXEL_FILL);
			updateScreen();
			break;
		case 2:
			//core0
			user = cpu0[0] - cpu0old[0];
			nice = cpu0[1] - cpu0old[1];
			system = cpu0[2] - cpu0old[2];
			idle = cpu0[3] - cpu0old[3];
			totalusage = user + nice + system;
			calc = 100 - (idle / (totalusage + idle) * 100);
			//core1
			user1 = cpu1[0] - cpu1old[0];
			nice1 = cpu1[1] - cpu1old[1];
			system1 = cpu1[2] - cpu1old[2];
			idle1 = cpu1[3] - cpu1old[3];
			totalusage1 = user1 + nice1 + system1;
			//overall
			usera = (cpu0[0] + cpu1[0]) - (cpu0old[0] + cpu1old[0]);
			nicea = (cpu0[1] + cpu1[1]) - (cpu0old[1] + cpu1old[1]);
			systema = (cpu0[2] + cpu1[2]) - (cpu0old[2] + cpu1old[2]);
			idlea = (cpu0[3] + cpu1[3]) - (cpu0old[3] + cpu1old[3]);
			totalusagea = usera + nicea + systema;
			calc = 100 - (idle / (totalusage + idle) * 100);
			calc1 = 100 - (idle1 / (totalusage1 + idle1) * 100);
			calca = 100 - (idlea / (totalusagea + idlea) * 100);
			targetpx = ceil(56 * calc/100)+20;
			targetpx1 = ceil(56 * calc1/100)+20;
			targetpxa = ceil(56 * calca/100)+20;
			g15r_drawLine(canvas, 20, 9, targetpx, 9, G15_PIXEL_FILL);
			g15r_drawLine(canvas, 20, 10, targetpx1, 10, G15_PIXEL_FILL);
			g15r_drawLine(canvas, 20, 11, targetpxa, 11, G15_PIXEL_FILL);
			updateScreen();
			break;
		case 3:
			//core0
			user = cpu0[0] - cpu0old[0];
			nice = cpu0[1] - cpu0old[1];
			system = cpu0[2] - cpu0old[2];
			idle = cpu0[3] - cpu0old[3];
			totalusage = user + nice + system;
			calc = 100 - (idle / (totalusage + idle) * 100);
			//core1
			user1 = cpu1[0] - cpu1old[0];
			nice1 = cpu1[1] - cpu1old[1];
			system1 = cpu1[2] - cpu1old[2];
			idle1 = cpu1[3] - cpu1old[3];
			totalusage1 = user1 + nice1 + system1;
			//core2
			usera = cpu2[0] - cpu2old[0];
			nicea = cpu2[1] - cpu2old[1];
			systema = cpu2[2] - cpu2old[2];
			idlea = cpu2[3] - cpu2old[3];
			totalusagea = usera + nicea + systema;
			calc = 100 - (idle / (totalusage + idle) * 100);
			calc1 = 100 - (idle1 / (totalusage1 + idle1) * 100);
			calca = 100 - (idlea / (totalusagea + idlea) * 100);
			targetpx = ceil(56 * calc/100)+20;
			targetpx1 = ceil(56 * calc1/100)+20;
			targetpxa = ceil(56 * calca/100)+20;
			g15r_drawLine(canvas, 20, 9, targetpx, 9, G15_PIXEL_FILL);
			g15r_drawLine(canvas, 20, 10, targetpx1, 10, G15_PIXEL_FILL);
			g15r_drawLine(canvas, 20, 11, targetpxa, 11, G15_PIXEL_FILL);
			updateScreen();
			break;
		}
		pthread_mutex_unlock( &LCDdrawMutex );
		msleep(500);
	}
	return (void*)NULL; //dirty and not needed but hey - it prevents warnings xD
}

void
*DrawLoadAverage (void *ptr)
{
	sg_load_stats *load_stat;
	char min5[MAX_LINE_LEN], min15[MAX_LINE_LEN];

	while(1) {
		if( (load_stat = sg_get_load_stats()) != NULL){
			sprintf(min5, "%.2f", load_stat->min5);
			sprintf(min15, "%.2f", load_stat->min15);
			pthread_mutex_lock( &LCDdrawMutex );
			clearArea(83, 28, 118, 43);
			g15r_ttfPrint(canvas, 86, 29, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, LEFT, (char *)"5M:");
			g15r_ttfPrint(canvas, 83, 36, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, LEFT, (char *)"15M:");
			g15r_ttfPrint(canvas, 102, 29, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, LEFT, min5);
			g15r_ttfPrint(canvas, 102, 36, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, LEFT, min15);
			updateScreen();
			pthread_mutex_unlock( &LCDdrawMutex );
		}


		sleep(10);
	}
	return (void*)NULL; //dirty and not needed but hey - it prevents warnings xD
}

void
*DrawMemoryUsage (void *ptr)
{
	sg_mem_stats *mem_stats;
	sg_swap_stats *swap_stats;

	long long used;
	int used_perc = 100, swap_used_perc = 100, targetpx = 20, targetpx2 = 20;

	while(1) {
		pthread_mutex_lock( &LCDdrawMutex );
		clearArea(20, 15, 76, 17);
		if( ((mem_stats=sg_get_mem_stats()) != NULL) && (swap_stats=sg_get_swap_stats()) != NULL){
			//cout << "Total: " << mem_stats->total/1024/1024 << " Used: " << mem_stats->used/1024/1024 << " Cached: " << mem_stats->cache/1024/1024 << " Free: " << mem_stats->free/1024/1024 << endl;
			used = mem_stats->used - mem_stats->cache;
			used_perc = ((unsigned int)used / ((unsigned int)mem_stats->total / 100));
			targetpx = ceil(56 * used_perc/100)+20;
			if(swap_stats->total > 0) { // <-- prevents a floating point exception, i learned this the hard way after i deleted my swap partition :P
				swap_used_perc = ((unsigned int)swap_stats->used / ((unsigned int)swap_stats->total / 100));
				targetpx2 = ceil(56 * swap_used_perc/100)+20;
			}
			//cout << "Used percent: " << used_perc << " Used value: " << used/1024/1024 << " Total: " << mem_stats->total/1024/1024 << " Targetpx: " << targetpx << endl;
			g15r_drawLine(canvas, 20, 15, targetpx, 15, G15_PIXEL_FILL);
			g15r_drawLine(canvas, 20, 16, targetpx, 16, G15_PIXEL_FILL);
			g15r_drawLine(canvas, 20, 17, targetpx2, 17, G15_PIXEL_FILL);
			updateScreen();
		}
		pthread_mutex_unlock( &LCDdrawMutex );
		msleep(500);
	}
	return (void*)NULL; //dirty and not needed but hey - it prevents warnings xD
}

void
*DrawFreeSpace (void *ptr)
{
	char df1free[MAX_LINE_LEN], df2free[MAX_LINE_LEN], df1cmd[MAX_LINE_LEN], df2cmd[MAX_LINE_LEN], line[MAX_LINE_LEN];
	float df1freeI, df2freeI;
	FILE *fpipe;

	while(1) {
		sprintf(df1cmd, "df | grep %s | awk '{print $4}'", DRIVE1);
		sprintf(df2cmd, "df | grep %s | awk '{print $4}'", DRIVE2);

		if ( !(fpipe = (FILE*)popen(df1cmd,"r")) ) { printf("\tProblems with pipe for getting drive 1 free space"); }
		if ( fscanf(fpipe, "%s", line) ) { sprintf(df1free, "%s", line); }
		pclose(fpipe);

		if ( !(fpipe = (FILE*)popen(df2cmd,"r")) ) { printf("\tProblems with pipe for getting drive 2 free space"); }
		if ( fscanf(fpipe, "%s", line) ) { sprintf(df2free, "%s", line); }
		pclose(fpipe);

		df1freeI = atof(df1free) / 1024 / 1024;
		df2freeI = atof(df2free) / 1024 / 1024;
		sprintf(df1free, "%.2fGB                               ", df1freeI);
		sprintf(df2free, "%.2fGB                               ", df2freeI);

		pthread_mutex_lock( &LCDdrawMutex );
		clearArea(3, 28, 36, 43);
		g15r_ttfPrint(canvas, 3, 29, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, RIGHT, df1free);
		g15r_ttfPrint(canvas, 3, 36, atoi(LCD_FONT_SIZE_SMALL), 1, G15_COLOR_BLACK, RIGHT, df2free);
		updateScreen();
		pthread_mutex_unlock( &LCDdrawMutex );
		sleep(30);
	}
	return (void*)NULL; //dirty and not needed but hey - it prevents warnings xD
}

void
*keyboard_watch(void *ptr) {
    unsigned int keystate;

    while(1) {
        recv(g15screen_fd,&keystate,4,0);
        if(keystate & G15_KEY_L1) {
        }
        else if(keystate & G15_KEY_L2) {
        	pthread_mutex_lock( &LCDdrawMutex );
        	g15r_ttfPrint(canvas, 45, 0, atoi(LCD_FONT_SIZE_MEDIUM), 0, G15_COLOR_BLACK, LEFT, (char *)"LCD1 pressed");
        	updateScreen();
        	pthread_mutex_unlock( &LCDdrawMutex );
            cycle=0;
        }
        else if(keystate & G15_KEY_L3) {
        	pthread_mutex_lock( &LCDdrawMutex );
			g15r_ttfPrint(canvas, 45, 0, atoi(LCD_FONT_SIZE_MEDIUM), 0, G15_COLOR_BLACK, LEFT, (char *)"LCD2 pressed");
			updateScreen();
			pthread_mutex_unlock( &LCDdrawMutex );
            cycle=1;
        }
        else if(keystate & G15_KEY_L4) {
        	pthread_mutex_lock( &LCDdrawMutex );
			g15r_ttfPrint(canvas, 45, 0, atoi(LCD_FONT_SIZE_MEDIUM), 0, G15_COLOR_BLACK, LEFT, (char *)"LCD3 pressed");
			updateScreen();
			pthread_mutex_unlock( &LCDdrawMutex );
			cycle=2;
        }
        else if(keystate & G15_KEY_L5) {
        	pthread_mutex_lock( &LCDdrawMutex );
			g15r_ttfPrint(canvas, 45, 0, atoi(LCD_FONT_SIZE_MEDIUM), 0, G15_COLOR_BLACK, LEFT, (char *)"LCD4 pressed");
			updateScreen();
			pthread_mutex_unlock( &LCDdrawMutex );
            cycle=3;
        }
        usleep(100*900);
    }
    return (void*)NULL; //dirty and not needed but hey - it prevents warnings xD
}

int
main (int argc, char *argv[])
{
	/* ----------------- some initialization stuff like reading the conf and lcd init ----------------- */
	connectToScreen ();
	g15r_clearScreen (canvas, 0);
	g15r_loadWbmpSplash(canvas, (char *)"./splash.wbmp");
	updateScreen();

	sg_init();

	read_config_var((char *)"mcdopes.conf", (char *)"MAX_UPSTREAM", MAX_UPSTREAM);
	read_config_var((char *)"mcdopes.conf", (char *)"MAX_DOWNSTREAM", MAX_DOWNSTREAM);
	read_config_var((char *)"mcdopes.conf", (char *)"CASETEMPCOMMAND", CASETEMPCOMMAND);
	read_config_var((char *)"mcdopes.conf", (char *)"CPUTEMPCOMMAND", CPUTEMPCOMMAND);
	read_config_var((char *)"mcdopes.conf", (char *)"LCD_FONT_SIZE_MEDIUM", LCD_FONT_SIZE_MEDIUM);
	read_config_var((char *)"mcdopes.conf", (char *)"LCD_FONT_SIZE_SMALL", LCD_FONT_SIZE_SMALL);
	read_config_var((char *)"mcdopes.conf", (char *)"LCD_FONT_MEDIUM", LCD_FONT_MEDIUM);
	read_config_var((char *)"mcdopes.conf", (char *)"LCD_FONT_SMALL", LCD_FONT_SMALL);
	read_config_var((char *)"mcdopes.conf", (char *)"NETIF", NETIF);
	read_config_var((char *)"mcdopes.conf", (char *)"DRIVE1", DRIVE1);
	read_config_var((char *)"mcdopes.conf", (char *)"DRIVE2", DRIVE2);
	read_config_var((char *)"mcdopes.conf", (char *)"CPUCORECOUNT", CPUCORECOUNT);

	g15r_ttfLoad(canvas, (char *)LCD_FONT_MEDIUM, atoi(LCD_FONT_SIZE_MEDIUM), 0);
	g15r_ttfLoad(canvas, (char *)LCD_FONT_SMALL, atoi(LCD_FONT_SIZE_SMALL), 1);

	sleep(4);
	/* ------------------------------------- all init stuff done -------------------------------------- */

	DrawGraphics(NULL);
	//pthread_create(&gfxthread, NULL, DrawGraphics, NULL);
	pthread_create(&clockthread, NULL, DrawClock, NULL);
	pthread_create(&tempthread, NULL, DrawTemps, NULL);
	pthread_create(&loadthread, NULL, DrawLoadAverage, NULL);
	pthread_create(&netthread, NULL, DrawNetworkTraffic, NULL);
	pthread_create(&cputhread, NULL, DrawCPUStats, NULL);
	pthread_create(&memthread, NULL, DrawMemoryUsage, NULL);
	pthread_create(&spcthread, NULL, DrawFreeSpace, NULL);
	pthread_create(&keywatch, NULL, keyboard_watch, NULL);

	pthread_join(clockthread, NULL);
	pthread_join(tempthread, NULL);
	pthread_join(loadthread, NULL);
	pthread_join(netthread, NULL);
	pthread_join(cputhread, NULL);
	pthread_join(memthread, NULL);
	pthread_join(gfxthread, NULL);
	pthread_join(spcthread, NULL);
	return 0;
}

int read_config_var( char *values_file, char *keyword , char value[] )
{
	static char str[MAX_LINE_LEN];
	int len;
	FILE * _file;

	if( keyword == NULL ) return (BAD);
	len = strlen(keyword);

	if( len > (MAX_LINE_LEN - 3)) return(UGLY);

	if( values_file ) {
		_file = fopen(values_file, "r");
		if (_file == NULL) return(BAD);
	}

	else { return(UGLY); }

	if( fseek(_file, 0, SEEK_SET) ) {
		fclose(_file);
		return(BAD);
	}

	char *warnPrevent;
	for(;;)
	{
		warnPrevent = fgets(str, MAX_LINE_LEN, _file);
		if( ferror(_file) || feof(_file) ) return(UGLY);
		len = strlen(str);

		if( strncmp(keyword, str, strlen(keyword)) == 0 ) {
				if (str[len - 1] == '\n') str[--len] = 0;
				sprintf(value, "%s", &str[strlen(keyword)+1] ); break;
		}
	}
	fclose(_file);
	return 0;
}  /* end of read_config_var */

int __nsleep(const struct timespec *req, struct timespec *rem)
{
    struct timespec temp_rem;
    if(nanosleep(req,rem)==-1) { __nsleep(rem,&temp_rem); }
    else { return 1; }
    return 2;
}

int msleep(unsigned long milisec)
{
    struct timespec req={0},rem={0};
    time_t sec=(int)(milisec/1000);
    milisec=milisec-(sec*1000);
    req.tv_sec=sec;
    req.tv_nsec=milisec*1000000L;
    __nsleep(&req,&rem);
    return 1;
}
